/**
* \file: hud.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* Sample application for osgStream capture library.
*
* \component: osgStream
*
* \author: Jens Georg <jgeorg@de.adit-jv.com>
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#include <sstream>
#include <memory>

#include <osg/ArgumentParser>
#include <osg/Camera>
#include <osg/MatrixTransform>
#include <osgViewer/Viewer>

#include "osgcamtogstbuf.h"
#include "demoscene.h"
#include "gstpipeline.h"

int main(int argc, char *argv[])
{
    std::string demo_path ="/opt/platform/cessna-osgstream.osg";
    uint width = 800;
    uint height = 480;
    uint samples = 0;
    std::string fpsIn = "25/1";
    std::string blocking_mode = "on";
    std::string use_oldbuffer = "no";
    uint numerator;
    uint denominator;

    osg::ArgumentParser arguments(&argc, argv);
    osg::ApplicationUsage *au = arguments.getApplicationUsage();

    au->setApplicationName(arguments.getApplicationName());
    au->setDescription(arguments.getApplicationName() + "is a demo for OpenSceneGraph to demonstrate H.264 RTP streaming of OSG scenes");
    au->addCommandLineOption("--demo_path <path>", "path with cessna");
    au->addCommandLineOption("--width <width>", "Width of the video stream");
    au->addCommandLineOption("--height <height>", "The height of the video stream");
    au->addCommandLineOption("--fps <FPS>", "The frame rate of the H.264 stream");
    au->addCommandLineOption("--samples <samples>", "The number of samples for multi-sampling");
    au->addCommandLineOption("--blockingRenderMode <renderMode>", "Stream Config for the application (on / off)");
    au->addCommandLineOption("--use_oldbuffer <use_oldbuffer>", "when rendering was stoped ,wheather to show old buffer or wait for rendering to start (yes / no) <default no>");
    au->addCommandLineOption("--format <format>", "format of the buffer");

    unsigned int helpType = 0;
    if ((helpType = arguments.readHelpType()))
    {
        arguments.getApplicationUsage()->write(std::cout, helpType);
        return 1;
    }

    // report any errors if they have occurred when parsing the program arguments.
    if (arguments.errors())
    {
        arguments.writeErrorMessages(std::cout);
        return 1;
    }

    std::string format ;
    if (arguments.read("--format", format))
    {
        if (format != "8888" && format != "565")
        {
            OSG_WARN << "Invalid color format, using 8888";
        }
    }

#if GST_CHECK_VERSION(1,0,0)
    format = "565";
#endif

    arguments.read("--width", width);
    arguments.read("--height", height);
    arguments.read("--blockingRenderMode", blocking_mode);
    arguments.read("--use_oldbuffer", use_oldbuffer);
    arguments.read("--demo_path", demo_path);
    arguments.read("--fps", fpsIn);


    generateTexGstbuf::BufferTraits bufTraits;
    if (format == "565")
    {
        bufTraits.red = 5;
        bufTraits.green = 6;
        bufTraits.blue = 5;
        bufTraits.alpha = 0;
    }
    else
    {
        bufTraits.red = 8;
        bufTraits.green = 8;
        bufTraits.blue = 8;
        bufTraits.alpha = 8;
    }

    osgCamtoGstbuf::OsgCamtoGstbuf::FboTraits fboTraits;
    // Enable depth buffer
    fboTraits.depth = 24;
    fboTraits.stencil = 8;

    bufTraits.config["PoolSize"] = "3";

    if (arguments.read("--samples", samples))
    {
        fboTraits.samples = samples;
    }

    // Set appsrc configuration
    osgGstpipeline::gstpipeline::appsrcTraits appsrc_traits;

    std::stringstream ss(fpsIn);
    char tmp;

    ss >> numerator;
    ss >> tmp;
    ss >> denominator;

    if (tmp == '/' && denominator > 0)
    {
        appsrc_traits.fpsNum = numerator;
        appsrc_traits.fpsDem = denominator;
    }
    else
    {
        OSG_WARN << "Invalid FPS fraction" << fpsIn << std::endl;
    }

    appsrc_traits.width = width;
    appsrc_traits.height = height;
    appsrc_traits.depth = bufTraits.red + bufTraits.green + bufTraits.blue + bufTraits.alpha ;

    if (format == "565")
    {
        appsrc_traits.endianness = 1234;
        appsrc_traits.red_mask = 0xF800;
        appsrc_traits.green_mask = 0x07E0;
        appsrc_traits.blue_mask = 0x001F;
        appsrc_traits.alpha_mask = 0x0000;
    }
    else
    {
        appsrc_traits.endianness = 4321;
        appsrc_traits.red_mask = 0xFF000000;
        appsrc_traits.green_mask = 0x00FF0000;
        appsrc_traits.blue_mask = 0x0000FF00;
        appsrc_traits.alpha_mask = 0x000000FF;
    }

    uint _width = width;
    uint _height = height;

    int loop = 1;

    osgViewer::Viewer viewer;
    viewer.setThreadingModel(osgViewer::Viewer::SingleThreaded);
    viewer.setReleaseContextAtEndOfFrameHint(false);

    while(loop < 30000)
    {
        width = _width;
        height = _height;
        // Create an instance of the encoder
        osgCamtoGstbuf::OsgCamtoGstbuf* encoder = new osgCamtoGstbuf::OsgCamtoGstbuf(bufTraits,fboTraits);
        encoder->setBuffersize(width,height);
        encoder->getBuffersize(&width,&height);

        osgGstpipeline::gstpipeline* pipeline_obj = new osgGstpipeline::gstpipeline(appsrc_traits);
        encoder->setBufferCallback(pipeline_obj ,pipeline_obj->buffercallback);
        pipeline_obj->set_stream_size(width,height);

        if(blocking_mode == "on")
        {
            pipeline_obj->blockMode = true;
        }
        else if(blocking_mode == "off")
        {
            pipeline_obj->blockMode = false;
        }
        else
        {
            OSG_WARN << "Invalid blockingRenderMode: " << blocking_mode <<
                    "\nDefault config selected: blockingRenderMode : on \n" <<std::endl;
        }

        if(use_oldbuffer == "yes")
        {
            pipeline_obj->use_oldbuffer = true;
        }
        else if(use_oldbuffer == "no")
        {
            pipeline_obj->use_oldbuffer = false;
        }
        else
        {
            OSG_WARN << "Invalid use_oldbuffer: " << blocking_mode <<
                    "\nDefault config selected: blockingRenderMode : no \n" <<std::endl;
        }
        pipeline_obj->start();

        // Get the camera from the encoder. You can modify whatever you want
        // _EXCEPT_ the pre- and post-draw callbacks.
        osg::ref_ptr<osg::Camera> radar = encoder->getOrCreateCamera();

        radar->setViewMatrix(osg::Matrixd::lookAt(osg::Vec3(0.0f, 0.0f, 120.0f),
                                                  osg::Vec3(),
                                                  osg::Y_AXIS));

        radar->setProjectionMatrix(osg::Matrixd::ortho2D(-120.0,
                                                          120.0,
                                                         -120.0,
                                                          120.0));

        osg::ref_ptr<DemoScene> scene = new DemoScene(demo_path);
        scene->ref();
        viewer.setSceneData(scene.get());
        viewer.setLightingMode(osg::View::SKY_LIGHT);
        viewer.realize();
        osg::GraphicsContext *ctx = viewer.getCamera()->getGraphicsContext();


        viewer.getCamera()->setViewMatrixAsLookAt(osg::Vec3(50,260,0),
                                                  osg::Vec3(50,0,0),
                                                  osg::Vec3(0,0,1) );

        radar->setGraphicsContext(ctx);

        // Add the camera as a slave camera to the viewer
        viewer.addSlave(radar, true);

#if !GST_CHECK_VERSION(1,0,0)
            /*
             * for rcar rezing not working need to check
             * maybe becuase of no proper release of buffers
             */
        int cnt = 10000;
#else
        int cnt = 10000;
#endif

        while( !viewer.done() && cnt)
        {
            // fire off the cull and draw traversals of the scene.
            viewer.frame();
            cnt--;

            if(cnt == 8000)
            {
                pipeline_obj->stop(); // pause the gstreamer pipeline
                width = 801;
                height = 601;
                encoder->setBuffersize(width,height);
                encoder->getBuffersize(&width,&height);
                pipeline_obj->set_stream_size(width,height);
                pipeline_obj->play();
            }

            if(cnt == 6000)
            {
                pipeline_obj->stop();
                width = 701;
                height = 551;
                encoder->setBuffersize(width,height);
                encoder->getBuffersize(&width,&height);
                pipeline_obj->set_stream_size(width,height);
                pipeline_obj->play(); // play the gstreamer pipeline
            }

            if(cnt == 4000)
            {
                pipeline_obj->stop(); // pause the gstreamer pipeline
                width = 601;
                height = 401;
                encoder->setBuffersize(width,height);
                encoder->getBuffersize(&width,&height);
                pipeline_obj->set_stream_size(width,height);
                pipeline_obj->play();
            }

            if(cnt == 2000)
            {
                pipeline_obj->stop();
                width = 501;
                height = 301;
                encoder->setBuffersize(width,height);
                encoder->getBuffersize(&width,&height);
                pipeline_obj->set_stream_size(width,height);
                pipeline_obj->play(); // play the gstreamer pipeline
            }
        }

        scene->unref();
        scene = NULL;
        encoder->detachFromCamera(radar);
        radar = NULL;
        encoder->setBufferCallback(NULL ,NULL);
        pipeline_obj->close();
        delete pipeline_obj;
#if !GST_CHECK_VERSION(1,0,0)
        /* needs to check for gen4
         * crashes when encoder deleted
         */
        delete encoder;
#endif
        std::cout<<"func "<<__func__<<"end of round "<<loop<<std::endl;
        loop++;
    }
    return 0;
}
